<?php
namespace hakoniwa\blocks\ssr\posts;

use hakoniwa\blocks\init\Define;
use hakoniwa\blocks\init\Kses;
use hakoniwa\blocks\util\Misc;

/**
 * 投稿一覧ブロック
 */
class Register {

	/**
	 * 初期設定
	 */
	public function __construct() {
		// オプションページ追加フック
		add_action( 'init', array( $this, 'register_block' ), 100 );
	}

	/**
	 * 出力（コールバック）
	 *
	 * @param array $attributes 属性
	 * @return string
	 */
	public function block_callback( $attributes ) {
		$should_load_enqueue_swiper = ! wp_script_is( Define::value('plugin_name') . '-swiper' );
		if ( $should_load_enqueue_swiper ) {
			Misc::enqueue_swiper();
		}

		$should_load_view_script = ! empty( $attributes['enableSlide'] ) && ! wp_script_is( Define::value( 'plugin_name' ) . '-posts-view-script' );
		if ( $should_load_view_script ) {
			wp_enqueue_script(
				Define::value( 'plugin_name' ) . '-posts-view-script'
			);			
		}

		// tax_queryを入れる
		$tax_query = array(
			'relation' => 'AND',
		);

		$term_data = array();

		$post_type  = $attributes['postType'];
		$taxonomies = get_object_taxonomies( $post_type, 'object' );

		foreach ( $taxonomies as $taxonomy ) {

			// Rest_base が指定してある時
			if ( null !== $taxonomy->rest_base && ! empty( $attributes[ $taxonomy->rest_base ] ) ) {
				$tax_query[] = array(
					'taxonomy' => $taxonomy->name,
					'field'    => 'term_id',
					'terms'    => $attributes[ $taxonomy->rest_base ],
				);

				$term_data[] = array(
					'taxonomy' => $taxonomy->name,
					'terms'    => $attributes[ $taxonomy->rest_base ],
				);
			} elseif ( null === $taxonomy->rest_base && ! empty( $attributes[ $taxonomy->name ] ) ) {
				$tax_query[] = array(
					'taxonomy' => $taxonomy->name,
					'field'    => 'term_id',
					'terms'    => $attributes[ $taxonomy->name ],
				);

				$term_data[] = array(
					'taxonomy' => $taxonomy->name,
					'terms'    => $attributes[ $taxonomy->name ],
				);
			}

			/*
			//if( array_key_exists( "$taxonomy->rest_base", $attributes ) && ! empty( $attributes[$taxonomy->rest_base] ) ){
			//	$tax_query[] = array(
			//		'taxonomy' => $taxonomy->rest_base,
			//		'field'    => 'term_id',
			//		'terms'    => $attributes[$taxonomy->rest_base],
			//	);
			//} else {
			//	$tax_query[] = array(
			//		'taxonomy' => $taxonomy->name,
			//		'field'    => 'term_id',
			//		'terms'    => $attributes[$taxonomy->name],
			//	);
			//}
			*/
		}

		$args = array(
			'post_type'           => $post_type,
			'ignore_sticky_posts' => true,
			'posts_per_page'      => $attributes['numberOfItems'],
			'post_status'         => 'publish',
			'tax_query'           => $tax_query,
			'orderby'             => $attributes['orderBy'],
			'order'               => $attributes['order'],
		);

		// var_dump($args);

		// 表示条件を変えるフィルター追加
		$args = apply_filters( Define::value( 'plugin_func_name' ) . '_posts_block_query', $args );

		// 投稿を取得
		$recent_posts = new \WP_Query( $args );

		if ( $recent_posts->have_posts() ) {

			// レイアウト別にテンプレート読み込み
			switch ( $attributes['layout'] ) {
				case 'list':
					return $this->list( $recent_posts, $attributes, $term_data );
					// break;
				default:
					return $this->grid( $recent_posts, $attributes, $term_data );
			}
		}
	}

	/**
	 * タクソノミー定義
	 *
	 * @return $result
	 */
	public function get_taxonomies() {
		$builtin_taxonomy = array(
			'categories',
			'tags',
		);
		$args             = array(
			'public'   => true,
			'_builtin' => false,

		);
		$output     = 'names'; // or objects
		$operator   = 'and'; // 'and' or 'or'
		$taxonomies = get_taxonomies( $args, $output, $operator );

		$result = array_merge( $builtin_taxonomy, $taxonomies );

		return $result;
	}

	/**
	 * ブロック登録
	 *
	 * @return void
	 */
	public function register_block() {

		$taxonomies = $this->get_taxonomies();

		$args = array(
			'numberOfItems'                      => array(
				'type'    => 'number',
				'default' => 4,
			),
			'postType'                           => array(
				'type'    => 'string',
				'default' => 'post',
			),
			'term_id'                            => array(
				'type'    => 'string',
				'default' => -1,
			),
			'textColor'                          => array(
				'type'    => 'string',
				'default' => '',
			),
			'columns'                            => array(
				'type'    => 'number',
				'default' => 2,
			),
			'defaultImage'                       => array(
				'type'    => 'string',
				'default' => '',
			),
			'displayThumbnail'                   => array(
				'type'    => 'boolean',
				'default' => true,
			),
			'displayAuthor'                      => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'displayDate'                        => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'displayModifiedDate'                => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'displayExcerpt'                     => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'displayNewMark'                     => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'displayUpdateMark'                  => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'displayLabel'                       => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'disablePostLink'                    => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'order'                              => array(
				'type'    => 'string',
				'default' => 'desc',
			),
			'orderBy'                            => array(
				'type'    => 'string',
				'default' => 'date',
			),
			'layout'                             => array(
				'type'    => 'string',
				'default' => 'grid',
			),
			'listLayoutType'                     => array(
				'type'    => 'string',
				'default' => '',
			),
			'dateLabel'                          => array(
				'type'    => 'string',
				'default' => '',
			),
			'modifiedDateLabel'                  => array(
				'type'    => 'string',
				'default' => '',
			),
			'thumbnailPosition'               => array(
				'type'    => 'string',
				'default' => 'top',
			),
			'thumbnailAspectRatio'               => array(
				'type'    => 'string',
				'default' => '16-9',
			),
			'newMarkText'                        => array(
				'type'    => 'string',
				'default' => 'NEW',
			),
			'newMarkDay'                         => array(
				'type'    => 'number',
				'default' => 1,
			),
			'newMarkColor'                       => array(
				'type'    => 'string',
				'default' => '',
			),
			'newMarkBackgroundColor'             => array(
				'type'    => 'string',
				'default' => '',
			),
			'newMarkBold'                        => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'updateMarkText'                     => array(
				'type'    => 'string',
				'default' => 'UPDATE',
			),
			'updateMarkDay'                      => array(
				'type'    => 'number',
				'default' => 1,
			),
			'updateMarkColor'                    => array(
				'type'    => 'string',
				'default' => '',
			),
			'updateMarkBackgroundColor'          => array(
				'type'    => 'string',
				'default' => '',
			),
			'updateMarkBold'                     => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'paddingTop'                         => array(
				'type' => 'number',
			),
			'paddingTopUnit'                     => array(
				'type'    => 'string',
				'default' => 'px',
			),
			'paddingBottom'                      => array(
				'type' => 'number',
			),
			'paddingBottomUnit'                  => array(
				'type'    => 'string',
				'default' => 'px',
			),

			'blockID'                            => array(
				'type'    => 'string',
				'default' => '',
			),
			'enableSlide'                        => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'direction'                          => array(
				'type'    => 'string',
				'default' => 'horizontal',
			),
			'speed'                              => array(
				'type'    => 'number',
				'default' => 0.3,
			),
			'spaceBetween'                       => array(
				'type'    => 'number',
				'default' => 0,
			),
			'autoplay'                           => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'autoplayDelay'                      => array(
				'type'    => 'number',
				'default' => 3,
			),
			'autoplayReverseDirection'           => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'autoplayDisableOnInteraction'       => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'effect'                             => array(
				'type'    => 'string',
				'default' => 'slide',
			),
			'scrollbar'                          => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'slidesPerView'                      => array(
				'type' => 'number',
			),
			'loop'                               => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'centeredSlides'                     => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'pagination'                         => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'paginationType'                     => array(
				'type'    => 'string',
				'default' => '',
			),
			'navigation'                         => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'responsive'                         => array(
				'type'    => 'boolean',
				'default' => false,
			),
			'breakPointTabletSlidesPreview'      => array(
				'type' => 'number',
			),
			'breakPointTabletSpaceBetween'       => array(
				'type' => 'number',
			),
			'breakPointTabletslidesPerGroup'     => array(
				'type' => 'number',
			),
			'breakPointSmartphoneSlidesPreview'  => array(
				'type' => 'number',
			),
			'breakPointSmartphoneSpaceBetween'   => array(
				'type' => 'number',
			),
			'breakPointSmartphoneslidesPerGroup' => array(
				'type' => 'number',
			),
			'slidesPerGroup'                     => array(
				'type' => 'number',
			),
			'scrollbarColor'                     => array(
				'type' => 'string',
			),
			'navigationColor'                    => array(
				'type' => 'string',
			),
			'paginationColor'                    => array(
				'type' => 'string',
			),
			'titleElement'                       => array(
				'type'    => 'string',
				'default' => 'h2',
			),
			'listBorderStyle'                    => array(
				'type'    => 'string',
				'default' => 'dotted',
			),
			'labelType'                          => array(
				'type'    => 'string',
				'default' => '',
			),
			'labelText'                          => array(
				'type'    => 'string',
				'default' => '',
			),
			'labelColor'                         => array(
				'type'    => 'string',
				'default' => '',
			),
			'labelBackgroundColor'               => array(
				'type'    => 'string',
				'default' => '',
			),
			'labelPosition'                      => array(
				'type'    => 'string',
				'default' => '',
			),
		);

		foreach ( $taxonomies as $taxonomy ) {
			$args[ $taxonomy ] = array(
				'type'    => 'array',
				'default' => array(),
				'items'   => array(
					'type' => 'integer',
				),
			);
		}

		register_block_type(
			Define::value( 'plugin_path' ) . 'build/posts',
			array(
				'attributes'      => $args,
				'render_callback' => array( $this, 'block_callback' ),
			)
		);
	}

	/**
	 * ラベル表示
	 *
	 * @param int   $attributes 属性
	 * @param array $term_data ラベルデータ
	 * @return mixed
	 */
	public function label_html( $attributes, $term_data ) {

		if ( ! $attributes['displayLabel'] ) {
			return false;
		}

		$label_name = '';
		$label_slug = '';
		$label_id   = '';

		$post_label_styles = array();

		if ( $attributes['labelColor'] ) {
			$post_label_styles[] = 'color:' . $attributes['labelColor'] . ';';
		}

		if ( $attributes['labelBackgroundColor'] ) {
			$post_label_styles[] = 'background-color:' . $attributes['labelBackgroundColor'] . ';';
		}

		$post_label_styles_output = ' style="' . implode( '', $post_label_styles ) . '"';

		if ( 'text' === $attributes['labelType'] ) {
			$label_name = $attributes['labelText'];
		} else {
			if ( ! empty( $term_data ) ) {

				// タームidが投稿に紐づけられていれば出力
				$block_term_ids = array();
				$post_term_ids  = array();

				foreach ( $term_data as $data ) {
					// termがあるかどうか確認
					$terms = wp_get_post_terms( get_the_ID(), $data['taxonomy'], array( 'fields' => 'ids' ) );

					if ( $terms ) {

						foreach ( $terms as $term ) {

							foreach ( $data['terms'] as $post_term ) {
								if ( $term == $post_term ) {
									$get_term_by = get_term_by( 'id', $term, $data['taxonomy'] );

									if ( $get_term_by ) {
										$label_name = $get_term_by->name;
										$label_slug = ' post-label-slug-' . $get_term_by->slug;
										$label_id   = ' post-label-id-' . $get_term_by->term_id;

										break;
									}
								}
							}

							if ( ! empty( $label_name ) ) {
								break;
							}
						}

						break;
					}
				}
			} else {
				return false;
			}
		}

		$output = '<span class="post-label' . $label_id . $label_slug . '"' . $post_label_styles_output . '>' . $label_name . '</span>';

		return $output;
	}

	/**
	 * サムネイル 表示
	 */
	public function thumbnail( $attributes, $post_id, $term_data ){
		if ( false === $attributes['displayThumbnail'] ) {
			return false;
		}

		$disable_post_link = $attributes['disablePostLink'];
		$is_post_link = apply_filters( Define::value( 'plugin_func_name' ) . '_is_post_link', true, $post_id );
		$noimageicon = '<svg xmlns="http://www.w3.org/2000/svg" class="icon icon-tabler icon-tabler-photo" width="24" height="24" viewBox="0 0 24 24" stroke-width="1.25" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round"><path stroke="none" d="M0 0h24v24H0z" fill="none"></path><line x1="15" y1="8" x2="15.01" y2="8"></line><rect x="4" y="4" width="16" height="16" rx="3"></rect><path d="M4 15l4 -4a3 5 0 0 1 3 0l5 5"></path><path d="M14 14l1 -1a3 5 0 0 1 3 0l2 2"></path></svg>';
		$output = '';

		if ( has_post_thumbnail( $post_id ) || $attributes['defaultImage'] ) {
			$noimage = '';
		} else {
			$noimage = ' noimage';
		}

		$aspect_ratio = ' aspect-ratio-' . $attributes['thumbnailAspectRatio'];
		$output      .= '<div class="post-image' . $aspect_ratio . $noimage . '">';

		if ( ! $disable_post_link && $is_post_link ) {
			$output .= '<a class="post-link" href="' . esc_url( get_permalink( $post_id ) ) . '">';
		}

		// ラベル
		$output .= $this->label_html( $attributes, $term_data );

		// サムネイル
		if ( '' === $noimage ) {
			$output .= '<figure>';
			if ( has_post_thumbnail( $post_id ) ) {
				$output .= get_the_post_thumbnail( $post_id, 'post-thumbnail', array( 'alt' => get_the_title() ) );
			} elseif ( $attributes['defaultImage'] ) {
				$output .= '<img src="' . esc_url( $attributes['defaultImage'] ) . '" alt="">';
			}
			$output .= '</figure>';
		} else {
			$output .= '<div class="no-image">' . $noimageicon . '</div>';
		}

		if ( $is_post_link ) {
			if ( ! $disable_post_link ) {
				$output .= '</a>';
			}
		}
		$output .= '</div>';

		return $output;
	}

	/**
	 * Detail 表示
	 */
	public function detail( $id, $attributes ) {
		$author = $attributes['displayAuthor'];

		$date       = $attributes['displayDate'];
		$date_label = '';
		if ( ! empty( $attributes['dateLabel'] ) ) {
			$date_label = $attributes['dateLabel'];
		}

		$modified_date       = $attributes['displayModifiedDate'];
		$modified_date_label = '';
		if ( ! empty( $attributes['modifiedDateLabel'] ) ) {
			$modified_date_label = $attributes['modifiedDateLabel'];
		}

		if ( $author || $date || $modified_date ) {
			$output .= '<footer class="has-2-xs-font-size">';
			if ( $date ) {
				$output .= '<time class="date publish-date" datetime="' . esc_attr( get_the_time( 'c' ) ) . '">' . esc_html( $date_label ) . esc_html( get_the_time( get_option( 'date_format' ) ) ) . '</time>';
			}
			if ( $modified_date ) {
				$output .= '<time class="date modified-date" datetime="' . esc_attr( get_the_modified_time( 'c' ) ) . '">' . esc_html( $modified_date_label ) . esc_html( get_the_modified_time( get_option( 'date_format' ) ) ) . '</time>';
			}
			if ( $author ) {
				$output .= '<span class="author">' . esc_html( get_the_author() ) . '</span>';
			}
			$output .= '</footer>';
		}
	}

	/**
	 * NEW マーク表示
	 *
	 * @param int   $id 投稿 ID
	 * @param array $attributes 属性
	 * @return mixed
	 */
	public function new_mark_html( $id, $attributes ) {
		if ( ! empty( $id ) && ! empty( $attributes ) && true === $attributes['displayNewMark'] && $attributes['newMarkText'] ) {

			$current_time = time();
			$timestamp    = get_post_time( 'U', true, $id ); // 投稿の時刻
			$new_mark     = $current_time - $timestamp; // 時刻の差分
			$date_limit   = 60 * 60 * 24 * $attributes['newMarkDay']; // 日間

			if ( $date_limit > $new_mark ) {
				$new_mark_class = array( 'mark', 'new-mark' );

				if ( $attributes['newMarkBold'] ) {
					$new_mark_class[] = 'has-mark-bold';
				}

				if ( $attributes['newMarkBackgroundColor'] ) {
					$new_mark_class[] = 'has-mark-background-color';
				}

				$new_mark_class_output = ' class="' . implode( ' ', $new_mark_class ) . '"';

				$new_mark_styles = array();

				if ( $attributes['newMarkColor'] ) {
					$new_mark_styles[] = 'color:' . $attributes['newMarkColor'] . ';';
				}

				if ( $attributes['newMarkBackgroundColor'] ) {
					$new_mark_styles[] = 'background-color:' . $attributes['newMarkBackgroundColor'] . ';';
				}

				$new_mark_styles_output = ' style="' . implode( '', $new_mark_styles ) . '"';

				$output = '<div' . $new_mark_class_output . '><span' . $new_mark_styles_output . '>' . $attributes['newMarkText'] . '</span></div>';

				return $output;
			} else {
				return false;
			}
		} else {
			return false;
		}
	}

	/**
	 * UPDATE マーク表示
	 *
	 * @param int   $id 投稿 ID
	 * @param array $attributes 属性
	 * @return mixed
	 */
	public function update_mark_html( $id, $attributes ) {
		if ( ! empty( $id ) && ! empty( $attributes ) && true === $attributes['displayUpdateMark'] && $attributes['updateMarkText'] ) {

			$current_time = time(); // 現在時刻
			$timestamp    = get_post_modified_time( 'U', true, $id ); // 投稿の時刻
			$update_mark  = $current_time - $timestamp; // 時刻の差分
			$date_limit   = 60 * 60 * 24 * $attributes['updateMarkDay']; // 日間

			if ( $date_limit > $update_mark ) {
				$update_mark_class = array( 'mark', 'update-mark' );

				if ( $attributes['updateMarkBold'] ) {
					$update_mark_class[] = 'has-mark-bold';
				}

				if ( $attributes['updateMarkBackgroundColor'] ) {
					$update_mark_class[] = 'has-mark-background-color';
				}

				$update_mark_class_output = ' class="' . implode( ' ', $update_mark_class ) . '"';

				$update_mark_styles = array();

				if ( $attributes['updateMarkColor'] ) {
					$update_mark_styles[] = 'color:' . $attributes['updateMarkColor'] . ';';
				}

				if ( $attributes['updateMarkBackgroundColor'] ) {
					$update_mark_styles[] = 'background-color:' . $attributes['updateMarkBackgroundColor'] . ';';
				}

				$update_mark_styles_output = ' style="' . implode( '', $update_mark_styles ) . '"';

				$output = '<div' . $update_mark_class_output . '><span' . $update_mark_styles_output . '>' . $attributes['updateMarkText'] . '</span></div>';

				return $output;
			} else {
				return false;
			}
		} else {
			return false;
		}
	}

	/**
	 * グリッドレイアウト
	 *
	 * @param wp_query $recent_posts 投稿
	 * @param array    $attributes 属性
	 * @param array    $term_data ラベルデータ
	 * @return mixed
	 */
	public function grid( $recent_posts, $attributes, $term_data = array() ) {
		$class             = '';
		$slider_class      = '';
		$json              = '';
		$padding_array     = array();
		$posts_styles      = '';
		$id                = 'block-' . $attributes['blockID'];
		$column            = ' columns-' . absint( $attributes['columns'] );
		$disable_post_link = $attributes['disablePostLink'];
		$textcolor         = $attributes['textColor'];
		$excerpt           = $attributes['displayExcerpt'];

		// タイトル属性
		if ( ! empty( $attributes['titleElement'] ) ) {
			$title_element = $attributes['titleElement'];
		}

		// スライダーを有効にしている場合
		if ( $attributes['enableSlide'] ) {

			$column       = '';
			$slider_class = ' class="swiper-slide"';

			$data = array(
				'blockID'   => $attributes['blockID'],
				'effect'    => $attributes['effect'],
				'direction' => $attributes['direction'],
			);

			if ( ! empty( $attributes['slidesPerView'] ) ) {
				$data['slidesPerView'] = $attributes['slidesPerView'];
			}

			if ( ! empty( $attributes['loop'] ) ) {
				$data['loop'] = $attributes['loop'];
			}

			// エフェクト「キューブ」のシャドウを無効
			if ( 'cube' === $attributes['effect'] ) {
				$data['cubeEffect'] = array(
					'shadow'       => false,
					'slideShadows' => false,
				);
			}

			// エフェクト「フリップ」のシャドウを無効
			if ( 'flip' === $attributes['effect'] ) {
				$data['flipEffect'] = array(
					'slideShadows' => false,
				);
			}

			// エフェクト「フェード」のクロスフェードを有効
			if ( 'fade' === $attributes['effect'] ) {
				$data['fadeEffect'] = array(
					'crossFade' => true,
				);
			}

			// エフェクト「カバーフロー」のシャドウを無効
			if ( 'coverflow' === $attributes['effect'] ) {
				$data['coverflowEffect'] = array(
					'slideShadows' => false,
				);
			}

			// エフェクト「スライド」「カバーフロー」のみ
			if ( 'slide' === $attributes['effect'] || 'coverflow' === $attributes['effect'] ) {
				// Spacebetween
				$data['spaceBetween'] = $attributes['spaceBetween'];

				// slidesPerView
				if ( ! empty( $attributes['slidesPerView'] ) ) {
					$data['slidesPerView'] = $attributes['slidesPerView'];
				}

				// CenterSlides
				if ( true === $attributes['centeredSlides'] ) {
					$data['centeredSlides'] = $attributes['centeredSlides'];
				}

				// Group
				if ( ! empty( $attributes['slidesPerGroup'] ) ) {
					$data['slidesPerGroup'] = $attributes['slidesPerGroup'];

					if ( true === $attributes['loop'] ) {
						$data['loopFillGroupWithBlank'] = $attributes['loop'];
					}
				}

				// Responsive
				if ( true === $attributes['responsive'] ) {

					// 681
					if ( ! empty( $attributes['breakPointSmartphoneSlidesPreview'] ) ) {
						$data['breakpoints'][681]['slidesPerView'] = $attributes['breakPointSmartphoneSlidesPreview'];
					}

					if ( ! empty( $attributes['breakPointSmartphoneSpaceBetween'] ) ) {
						$data['breakpoints'][681]['spaceBetween'] = $attributes['breakPointSmartphoneSpaceBetween'];
					}

					if ( ! empty( $attributes['breakPointSmartphoneslidesPerGroup'] ) ) {
						$data['breakpoints'][681]['slidesPerGroup'] = $attributes['breakPointSmartphoneslidesPerGroup'];
					}

					// 769
					if ( ! empty( $attributes['breakPointTabletSlidesPreview'] ) ) {
						$data['breakpoints'][769]['slidesPerView'] = $attributes['breakPointTabletSlidesPreview'];
					}

					if ( ! empty( $attributes['breakPointTabletSpaceBetween'] ) ) {
						$data['breakpoints'][769]['spaceBetween'] = $attributes['breakPointTabletSpaceBetween'];
					}

					if ( ! empty( $attributes['breakPointTabletslidesPerGroup'] ) ) {
						$data['breakpoints'][769]['slidesPerGroup'] = $attributes['breakPointTabletslidesPerGroup'];
					}
				}
			}

			// Speed
			$data['speed'] = $attributes['speed'] * 1000;

			// Direction
			$data['direction'] = $attributes['direction'];

			// Autoplay
			if ( true === $attributes['autoplay'] ) {
				$data['autoplay']['delay'] = $attributes['autoplayDelay'] * 1000;

				if ( true === $attributes['autoplayReverseDirection'] ) {
					$data['autoplay']['reverseDirection'] = $attributes['autoplayReverseDirection'];
				}

				if ( false === $attributes['autoplayDisableOnInteraction'] ) {
					$data['autoplay']['disableOnInteraction'] = $attributes['autoplayDisableOnInteraction'];
				}
			}

			// Scrollbar
			if ( true === $attributes['scrollbar'] ) {
				$data['scrollbar']['el'] = '.swiper-scrollbar';
			}

			// Navigation
			if ( true === $attributes['navigation'] ) {
				$data['navigation']['nextEl'] = '.swiper-button-next';
				$data['navigation']['prevEl'] = '.swiper-button-prev';
			}

			if ( false === $attributes['scrollbar'] && 'progress' !== $attributes['paginationType'] && 'flip' !== $attributes['effect'] ) {
				// Loop
				if ( true === $attributes['loop'] ) {
					$data['loop'] = $attributes['loop'];
				}
			}

			// Pagination
			if ( true === $attributes['pagination'] ) {
				$data['pagination']['el']        = '.swiper-pagination';
				$data['pagination']['clickable'] = true;

				if ( 'dynamic' === $attributes['paginationType'] ) {
					$data['pagination']['dynamicBullets'] = true;
				}

				if ( 'progress' === $attributes['paginationType'] ) {
					$data['pagination']['type'] = 'progressbar';
				}

				if ( 'fraction' === $attributes['paginationType'] ) {
					$data['pagination']['type'] = 'fraction';
				}
			}

			// 最後に属性作る
			$json = htmlspecialchars( wp_json_encode( $data ), ENT_QUOTES );
		}

		//クラス名などの属性を作成
		if ( ! $attributes['enableSlide'] ) {
			$class .= 'articles';

			if ( isset( $attributes['align'] ) ) {
				if( 'wide' === $attributes['align'] ){
					$class .= ' alignwide';
				}
			}

			if ( isset( $attributes['layout'] ) ) {
				$class .= ' is-' . $attributes['layout'];
			}
	
			if ( isset( $attributes['columns'] ) && $attributes['columns'] > 1 ) {
				$class .= ' flex gap-8 columns-' . absint( $attributes['columns'] );
			}
	
			if ( isset( $attributes['thumbnailPosition'] ) ) {
				$class .= ' featured-image-' . $attributes['thumbnailPosition'];
			}
	
			if ( isset( $attributes['displayThumbnail'] ) ) {
				$class .= ' has-featured-image';
			}
		} else {
			$class .= 'is-slider';
		}

		// テキストカラー
		if ( $textcolor ) {
			$style = ' color:' . esc_attr( $textcolor );
		} else {
			$style = '';
		}

		// パディング
		if ( ! empty( $attributes['paddingTop'] ) ) {
			$padding_array['padding-top'] = $attributes['paddingTop'] . $attributes['paddingTopUnit'];
		}

		if ( ! empty( $attributes['paddingBottom'] ) ) {
			$padding_array['padding-bottom'] = $attributes['paddingBottom'] . $attributes['paddingBottomUnit'];
		}

		if ( ! empty( $padding_array ) ) {
			$posts_styles .= ' style="';
			foreach ( $padding_array as $attr => $value ) {
				$posts_styles .= $attr . ':' . $value . ';';
			}
			$posts_styles .= '"';
		}

		$wrapper_attributes = get_block_wrapper_attributes( 
			array(
				'style' => $style,
				'class' => $class,
				'id'    => $id,
			)
		);

		if( $json ){
			$json = ' data-slider="' . $json . '" ';
		}

		$output = '<div ' . $json . $wrapper_attributes . '>';

		if ( $attributes['enableSlide'] ) {
			$output .= '<div class="swiper-wrapper">';
		}

		while ( $recent_posts->have_posts() ) {
			$recent_posts->the_post();

			$is_post_link = apply_filters( Define::value( 'plugin_func_name' ) . '_is_post_link', true, get_the_ID() );

			$output .= '<article' . $slider_class . '>';
			$output .= $this->thumbnail( $attributes, get_the_ID(), $term_data );
			$output .= '<div class="post-detail" ' . $style . '>';
			if ( ! $disable_post_link && $is_post_link ) {
				$output .= '<a class="post-link" href="' . esc_url( get_permalink( get_the_ID() ) ) . '">';
			}

			$output .= '<header>';
			$output .= $this->new_mark_html( get_the_ID(), $attributes );
			$output .= $this->update_mark_html( get_the_ID(), $attributes );
			$output .= '<' . $title_element . ' class="post-title has-sm-font-size">' . get_the_title( get_the_ID() ) . '</' . $title_element . '>';
			$output .= '</header>';

			if ( $excerpt && get_the_excerpt() ) {
				$excerpt_html = '<div class="excerpt has-2-xs-font-size">' . get_the_excerpt() . '</div>';
				$output      .= apply_filters( Define::value( 'plugin_func_name' ) . '_excerpt_html', $excerpt_html, get_the_ID() );
			}
			if ( ! $disable_post_link && $is_post_link ) {
				$output .= '</a>';
			}

			$author = $attributes['displayAuthor'];

			$date       = $attributes['displayDate'];
			$date_label = '';
			if ( ! empty( $attributes['dateLabel'] ) ) {
				$date_label = $attributes['dateLabel'];
			}

			$modified_date       = $attributes['displayModifiedDate'];
			$modified_date_label = '';
			if ( ! empty( $attributes['modifiedDateLabel'] ) ) {
				$modified_date_label = $attributes['modifiedDateLabel'];
			}

			if ( $author || $date || $modified_date ) {
				$output .= '<footer class="has-2-xs-font-size">';
				if ( $date ) {
					$output .= '<time class="date publish-date" datetime="' . esc_attr( get_the_time( 'c' ) ) . '">' . esc_html( $date_label ) . esc_html( get_the_time( get_option( 'date_format' ) ) ) . '</time>';
				}
				if ( $modified_date ) {
					$output .= '<time class="date modified-date" datetime="' . esc_attr( get_the_modified_time( 'c' ) ) . '">' . esc_html( $modified_date_label ) . esc_html( get_the_modified_time( get_option( 'date_format' ) ) ) . '</time>';
				}
				if ( $author ) {
					$output .= '<span class="author">' . esc_html( get_the_author() ) . '</span>';
				}
				$output .= '</footer>';
			}
			$output .= '</div>';
			$output .= '</article>';

		}

		if ( $attributes['enableSlide'] ) {
			$output .= '</div>';

			if ( true === $attributes['pagination'] ) {
				$attrs       = '';
				$attrs_array = array(
					'class' => 'swiper-pagination',
				);

				if ( ! empty( $attributes['paginationColor'] ) ) {
					$attrs_array['data-pagination-color'] = $attributes['paginationColor'];
				}

				foreach ( $attrs_array as $attr => $value ) {
					$attrs .= ' ' . $attr . '="' . $value . '"';
				}

				$output .= '<div' . $attrs . '></div>';
			}

			if ( true === $attributes['navigation'] ) {
				$attrs       = '';
				$attrs_array = array(
					'class' => 'swiper-navigation',
				);

				if ( ! empty( $attributes['navigationColor'] ) ) {
					$attrs_array['data-navigation-color'] = $attributes['navigationColor'];
				}

				foreach ( $attrs_array as $attr => $value ) {
					$attrs .= ' ' . $attr . '="' . $value . '"';
				}

				$output .= '<div' . $attrs . '>';
				$output .= '<div class="swiper-button-next"></div>';
				$output .= '<div class="swiper-button-prev"></div>';
				$output .= '</div>';
			}

			if ( true === $attributes['scrollbar'] ) {
				$attrs       = '';
				$attrs_array = array(
					'class' => 'swiper-scrollbar',
				);

				if ( ! empty( $attributes['scrollbarColor'] ) ) {
					$attrs_array['data-scrollbar-color'] = $attributes['scrollbarColor'];
				}

				foreach ( $attrs_array as $attr => $value ) {
					$attrs .= ' ' . $attr . '="' . $value . '"';
				}

				$output .= '<div' . $attrs . '></div>';
			}
		}

		$output .= '</div>';

		wp_reset_postdata();
		
		return wp_kses_post( $output );
	}

	/**
	 * リスト（日付）
	 *
	 * @param wp_query $recent_posts 投稿
	 * @param array    $attributes 属性
	 * @param array    $term_data ラベルデータ
	 * @return string
	 */
	public function list( $recent_posts, $attributes, $term_data = array() ) {
		$class_name        = '';
		$json              = '';
		$padding_array     = array();
		$posts_styles      = '';
		$id                = ' id="posts-' . $attributes['blockID'] . '"';
		$column            = ' columns-' . absint( $attributes['columns'] );
		$disable_post_link = $attributes['disablePostLink'];
		$textcolor         = $attributes['textColor'];
		$excerpt           = $attributes['displayExcerpt'];
		$border            = $attributes['listBorderStyle'];

		// タイトル属性
		if ( ! empty( $attributes['titleElement'] ) ) {
			$title_element = $attributes['titleElement'];
		}

		// パディング
		if ( ! empty( $attributes['paddingTop'] ) ) {
			$padding_array['padding-top'] = $attributes['paddingTop'] . $attributes['paddingTopUnit'];
		}

		if ( ! empty( $attributes['paddingBottom'] ) ) {
			$padding_array['padding-bottom'] = $attributes['paddingBottom'] . $attributes['paddingBottomUnit'];
		}

		if ( ! empty( $padding_array ) ) {
			$posts_styles .= ' style="';
			foreach ( $padding_array as $attr => $value ) {
				$posts_styles .= $attr . ':' . $value . ';';
			}
			$posts_styles .= '"';
		}

		//クラス名などの属性を作成
		$class = 'articles';

		if ( isset( $attributes['layout'] ) ) {
			$class .= ' is-' . $attributes['layout'];
		}

		if ( isset( $attributes['listLayoutType'] ) && '' !== $attributes['listLayoutType'] ) {
			$class .= ' is-list-' . $attributes['listLayoutType'];
		}

		// ボーダースタイル
		if ( 'none' !== $border ) {
			$class .= ' is-border is-border-' . esc_attr( $border );
		} else {
			$class .= '';
		}

		if ( isset( $attributes['thumbnailPosition'] ) && 'date' !== $attributes['listLayoutType'] ) {
			$class .= ' featured-image-' . $attributes['thumbnailPosition'];
		}

		if ( isset( $attributes['displayThumbnail'] ) && 'date' !== $attributes['listLayoutType'] ) {
			$class .= ' has-featured-image';
		}

		// テキストカラー
		if ( $textcolor ) {
			$style = ' color:' . esc_attr( $textcolor );
		} else {
			$style = '';
		}

		$wrapper_attributes = get_block_wrapper_attributes( 
			array( 
				'style' => $style,
				'class' => $class,
			)
		);

		$output = '<div ' . $wrapper_attributes . '>';
		
		while ( $recent_posts->have_posts() ) {
			$recent_posts->the_post();

			$is_post_link = apply_filters( Define::value( 'plugin_func_name' ) . '_is_post_link', true, get_the_ID() );

			$output .= '<article>';

			if( $attributes['listLayoutType'] !== 'date' ){
				$output .= $this->thumbnail( $attributes, get_the_ID(), $term_data );
			}

			$author = $attributes['displayAuthor'];

			$date       = $attributes['displayDate'];
			$date_label = '';
			if ( ! empty( $attributes['dateLabel'] ) ) {
				$date_label = $attributes['dateLabel'];
			}

			$modified_date       = $attributes['displayModifiedDate'];
			$modified_date_label = '';
			if ( ! empty( $attributes['modifiedDateLabel'] ) ) {
				$modified_date_label = $attributes['modifiedDateLabel'];
			}

			if( $attributes['listLayoutType'] === 'date' ){
				if ( $date ) {
					$output .= '<div class="post-date">';
					$output .= '<time class="date publish-date" datetime="' . esc_attr( get_the_time( 'c' ) ) . '">' . esc_html( $date_label ) . esc_html( get_the_time( get_option( 'date_format' ) ) ) . '</time>';
					$output .= '</div>';
				}
	
				if ( $modified_date ) {
					$output .= '<div class="post-date">';
					$output .= '<time class="date modified-date" datetime="' . esc_attr( get_the_modified_time( 'c' ) ) . '">' . esc_html( $modified_date_label ) . esc_html( get_the_modified_time( get_option( 'date_format' ) ) ) . '</time>';
					$output .= '</div>';
				}

				// ラベル
				$output .= $this->label_html( $attributes, $term_data );
			}

			$output .= '<div class="post-detail">';

			if ( ! $disable_post_link && $is_post_link ) {
				$output .= '<a class="post-link" href="' . esc_url( get_permalink( get_the_ID() ) ) . '">';
			}

			$output .= '<header>';
			$output .= $this->new_mark_html( get_the_ID(), $attributes );
			$output .= $this->update_mark_html( get_the_ID(), $attributes );
			$output .= '<' . $title_element . ' class="post-title has-sm-font-size">' . get_the_title( get_the_ID() ) . '</' . $title_element . '>';
			$output .= '</header>';
			if ( $excerpt && get_the_excerpt() ) {
				$excerpt_html = '<div class="excerpt has-2-xs-font-size">' . get_the_excerpt() . '</div>';
				$output      .= apply_filters( Define::value( 'plugin_func_name' ) . '_excerpt_html', $excerpt_html, get_the_ID() );
			}
			if ( ! $disable_post_link && $is_post_link ) {
				$output .= '</a>';
			}

			if( $attributes['listLayoutType'] === 'date' ){
				if ( $author ) {
					$output .= '<footer>';
					if ( $author ) {
						$output .= '<span class="author">' . esc_html( get_the_author() ) . '</span>';
					}
					$output .= '</footer>';
				}
			} else {
				if ( $author || $date || $modified_date ) {
					$output .= '<footer class="has-2-xs-font-size">';
					if ( $date ) {
						$output .= '<time class="date publish-date" datetime="' . esc_attr( get_the_time( 'c' ) ) . '">' . esc_html( $date_label ) . esc_html( get_the_time( get_option( 'date_format' ) ) ) . '</time>';
					}
					if ( $modified_date ) {
						$output .= '<time class="date modified-date" datetime="' . esc_attr( get_the_modified_time( 'c' ) ) . '">' . esc_html( $modified_date_label ) . esc_html( get_the_modified_time( get_option( 'date_format' ) ) ) . '</time>';
					}
					if ( $author ) {
						$output .= '<span class="author">' . esc_html( get_the_author() ) . '</span>';
					}
					$output .= '</footer>';
				}			
			}

			$output .= '</div>';
			$output .= '</article>';

		}

		$output .= '</div>';

		wp_reset_postdata();

		return wp_kses_post( $output );
	}
}

use hakoniwa\blocks\ssr\posts\Register;
new Register();