<?php
namespace hakoniwa\tools\editor\metabox;

use hakoniwa\tools\init\Define;

class TOC {

	/**
	 * Constructor.
	 */
	public function __construct() {

		// Metabox追加
		add_action( 'add_meta_boxes', [ $this, 'add_meta_box' ], 10 );

		add_action( 'save_post', [ $this, 'save_meta_box' ], 10 );

	}

	/**
	 * Metabox Name. 
	 */
	public function meta_box_name(){
		return Define::value( 'plugin_func_name' ) . '_meta_box_toc';
	}

	/**
	 * Adds a box to the main column on the Post and Page edit screens.
	 */
	public function add_meta_box() {

		$screens = $this->get_all_post_types();
		foreach ( $screens as $screen ) {
			add_meta_box(
				$this->meta_box_name() . '_' . $screen,
				__( 'Table of contents', 'hakoniwa-tools' ),
				array( $this, 'meta_box_callback' ),
				$screen,
				'side'
			);
		}
	}

	/**
	 * Prints the box content.
	 *
	 * @param WP_Post $post The object for the current post/page.
	 */
	public function meta_box_callback( $post ) {

		// Add a nonce field so we can check for it later.
		wp_nonce_field( $this->meta_box_name() . '_save', $this->meta_box_name() . '_nonce' );

		/*
			* Use get_post_meta() to retrieve an existing value
			* from the database and use the value for the form.
			*/
		$value = get_post_meta( get_the_ID(), '_' . $this->meta_box_name() . '_' . get_post_type(), true );

		$checked = ( $value ) ? ' checked="checked"' : '';

		echo '<label for="' . esc_attr( $this->meta_box_name() ) . '"><input type="checkbox" id="' . esc_attr( $this->meta_box_name() ) . '" name="' . esc_attr( $this->meta_box_name() ) . '_new_field" value="1" ' . esc_attr( $checked ) . ' />目次を非表示</label>';
	}

	/**
	 * When the post is saved, saves our custom data.
	 *
	 * @param int $post_id The ID of the post being saved.
	 */
	public function save_meta_box( $post_id ) {

		// Check if our nonce is set.
		if ( ! isset( $_POST[ $this->meta_box_name() . '_nonce' ] ) ) {
			return;
		}

		// Verify that the nonce is valid.
		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST[ $this->meta_box_name() . '_nonce' ] ) ), $this->meta_box_name() . '_save' ) ) {
			return;
		}

		// If this is an autosave, our form has not been submitted, so we don't want to do anything.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check the user's permissions.
		if ( isset( $_POST['post_type'] ) && 'page' === $_POST['post_type'] ) {

			if ( ! current_user_can( 'edit_page', $post_id ) ) {
				return;
			}
		} else {

			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				return;
			}
		}

		/* OK, it's safe for us to save the data now. */

		$value = get_post_meta( $post_id, '_' . $this->meta_box_name() . '_' . get_post_type(), true );

		// 値がポストされていなかったら
		if ( ! isset( $_POST[ $this->meta_box_name() . '_new_field' ] ) ) {
			if ( $value ) {
				delete_post_meta( $post_id, '_' . $this->meta_box_name() . '_' . get_post_type(), 1 );
			}
			return;
		} else {
			// Sanitize user input.
			$my_data = sanitize_text_field( wp_unslash( $_POST[ $this->meta_box_name() . '_new_field' ] ) );

			// Update the meta field in the database.
			update_post_meta( $post_id, '_' . $this->meta_box_name() . '_' . get_post_type(), $my_data );
		}
	}

	/**
	 * 投稿タイプ
	 */
	public function get_all_post_types() {
		$args     = array(
			'public' => true,
		);
		$output   = 'names';
		$operator = 'and';
		$pts      = get_post_types( $args, $output, $operator );

		return $pts;
	}

}

use hakoniwa\tools\editor\metabox;
new TOC();