<?php
namespace hakoniwa\tools\util;

use hakoniwa\tools\init\Define;
use hakoniwa\theme\util\Functions;

class DefaultOptions {

	/**
	 * Default Settings
	 *
	 * @var array
	 */
	public static function general_settings(){
		return array(
			'content_size' => '640px',
			'wide_size'    => '960px',
			'scroll_to_button' => [
				'desktop' => 1,
				'tablet' => 1,
				'smartphone' => 1,
			],
		);
	}

	/**
	 * デバッグログ出力
	 *
	 * @param string $message
	 * @param mixed $data
	 * @return void
	 */
	private static function debug_log( $message, $data = null ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG === true ) {
			$log_message = '[DefaultOptions] ' . $message;
			if ( $data !== null ) {
				$log_message .= ' | Data: ' . print_r( $data, true );
			}
			error_log( $log_message );
		}
	}

	/**
	 * 既存設定を完全に保持（ネストした配列のみ再帰処理）
	 * ユーザーが一度でも保存した設定には、新規キーを追加しない
	 *
	 * @param array $default デフォルト値
	 * @param array $current 既存値（ユーザー設定）
	 * @return array
	 */
	private static function preserve_user_settings( array $default, array $current ) {
		// 既存値をそのまま保持
		$merged = $current;

		foreach ( $current as $key => $value ) {
			if ( is_array( $value ) && isset( $default[$key] ) && is_array( $default[$key] ) ) {
				// 両方が配列の場合は再帰的に処理
				self::debug_log( "  Recursively preserving key '{$key}'" );
				$merged[$key] = self::preserve_user_settings( $default[$key], $value );
			} else {
				// 既存値をそのまま保持
				self::debug_log( "  Preserving existing key '{$key}'", $value );
			}
		}

		self::debug_log( "  User settings completely preserved, no new keys added" );
		return $merged;
	}

	/**
	 * プラグイン有効後 初期設定
	 *
	 * @return void
	 */
	public static function setting_options() {
		
		self::debug_log( '========== setting_options START ==========' );
		self::debug_log( 'WP_DEBUG status', defined( 'WP_DEBUG' ) ? WP_DEBUG : 'not defined' );
		
		// テーマオプション名を取得
		$theme_options_name = Define::value( 'theme_options_name' );
		self::debug_log( 'Theme options name', $theme_options_name );

		$default_options = [
			'general' => self::general_settings(),
			'meta' => [
				'active' => [
					'ogp' => 1,
					'twitter' => 1,
					'facebook' => 1,
				],
				'twitter_card_type' => 'summary_large_image',
			],
			'schema_org' => [
				'post_type' => [
					'post' => 1,
					'page' => 1,
				],
				'page_type' => 'BlogPosting',
			],
			'toc' => [
				'toc_enable' => 1,
				'toc_fixed_enable' => 1,
				'toc_heading_level' => 3,
				'toc_heading_level_number' => 4,
				'toc_label' => __( 'Table of Contents', 'hakoniwa-tools' ),
			]
		];

		foreach ( $default_options as $key => $value ) {
			$option_name = $theme_options_name . '_' . $key;
			
			self::debug_log( "---------- Processing: {$key} ----------" );
			self::debug_log( "Option name", $option_name );
			
			$current_value = get_option( $option_name, false );
			
			self::debug_log( "Current value exists", $current_value !== false ? 'YES' : 'NO' );
			
			if ( $current_value !== false ) {
				self::debug_log( "Current value type", gettype( $current_value ) );
				self::debug_log( "Current value", $current_value );
			}
			
			self::debug_log( "Default value", $value );

			if ( is_array( $value ) ) {
				// 配列オプションの場合
				if ( $current_value === false ) {
					// オプションが存在しない場合は新規作成
					self::debug_log( "Action", "Creating new option (add_option) - first time setup" );
					$result = add_option( $option_name, $value, '', 'no' );
					self::debug_log( "add_option result", $result ? 'SUCCESS' : 'FAILED' );
					
					// 追加後の値を確認
					$verify_value = get_option( $option_name, false );
					self::debug_log( "Verified value after add_option", $verify_value );
					
				} else {
					// 既存のオプションがある場合は完全に保持
					self::debug_log( "Action", "Preserving existing user settings (no new keys added)" );
					
					// 配列であることを確保
					$current_value = is_array( $current_value ) ? $current_value : [];
					self::debug_log( "Current value (ensured array)", $current_value );
					
					// ユーザー設定を完全保持（ネストした配列のみ再帰処理）
					$merged_value = self::preserve_user_settings( $value, $current_value );
					self::debug_log( "Preserved value", $merged_value );
					
					// 配列の比較（シリアライズして比較）
					$is_different = ( serialize( $merged_value ) !== serialize( $current_value ) );
					self::debug_log( "Values are different", $is_different ? 'YES' : 'NO' );
					
					// 実際には既存設定を保持するだけなので、差分は発生しないはず
					if ( $is_different ) {
						self::debug_log( "Action", "Updating option (update_option)" );
						$result = update_option( $option_name, $merged_value );
						self::debug_log( "update_option result", $result ? 'SUCCESS' : 'FAILED' );
						
						// 更新後の値を確認
						$verify_value = get_option( $option_name, false );
						self::debug_log( "Verified value after update_option", $verify_value );
					} else {
						self::debug_log( "Action", "No update needed (user settings preserved as-is)" );
					}
				}
			} else {
				// 通常のオプション(オプションが存在しない場合のみ追加)
				if ( $current_value === false ) {
					self::debug_log( "Action", "Creating new scalar option (add_option)" );
					$result = add_option( $option_name, $value, '', 'no' );
					self::debug_log( "add_option result", $result ? 'SUCCESS' : 'FAILED' );
					
					// 追加後の値を確認
					$verify_value = get_option( $option_name, false );
					self::debug_log( "Verified value after add_option", $verify_value );
				} else {
					self::debug_log( "Action", "Scalar option already exists, preserving user setting" );
				}
			}
		}
		
		self::debug_log( '========== setting_options END ==========' );
	}
}