<?php
namespace hakoniwa\tools\options;

use hakoniwa\tools\init\Define;
use hakoniwa\tools\util\CreateForm;

class Analytics {

	/**
	 * Constructor.
	 */
	public function __construct() {
		// オプションページ追加フック
		add_action( 'admin_menu', [ $this, 'register_settings' ], 10 );

		add_action( 'wp_head', [ $this, 'add_analytics_tag' ], 10 );

		add_action( 'wp_head', [ $this, 'add_gtm_header' ], 10 );
		add_action( 'wp_body_open', [ $this, 'add_gtm_body' ], 10 );

	}

	/**
	 * ページ名
	 */
	public function page_name(){
		return Define::value( 'theme_options_name' ) . '_analytics';
	}

	/**
	 * フォーム追加
	 */
	public function register_settings() {
		$create_form = new CreateForm;

		register_setting(
			$this->page_name(),
			$this->page_name(),
			array( $this, 'sanitize' )
		);

		add_settings_section(
			$this->page_name(),
			'',
			array( $create_form, 'nonce' ),
			$this->page_name()
		);

		add_settings_field(
			'analytics_range',
			__( 'Include logged-in users in the measurement range', 'hakoniwa-tools' ),
			array( $create_form, 'checkbox' ),
			$this->page_name(),
			$this->page_name(),
			array(
				'title'       => _x( 'Enable', 'Option:analytics', 'hakoniwa-tools' ),
				'label'       => 'analytics_range',
				'page_name'   => $this->page_name(),
				'description' => '',
			)
		);

		add_settings_field(
			'analytics_gtm',
			'Google Tag Manager',
			array( $create_form, 'text' ),
			$this->page_name(),
			$this->page_name(),
			array(
				'title'       => '',
				'label'       => 'analytics_gtm',
				'page_name'   => $this->page_name(),
				'description' => __( 'Put gtm container id in the form GTM-XXXXXXX.', 'hakoniwa-tools' ),
			)
		);

		add_settings_field(
			'analytics_gtag',
			'Google Analytics 4',
			array( $create_form, 'text' ),
			$this->page_name(),
			$this->page_name(),
			array(
				'title'       => '',
				'label'       => 'analytics_gtag',
				'page_name'   => $this->page_name(),
				'description' => __( 'Put gtag in the form G-XXXXXXXXXXXX.Leave blank if you have already set up Google Tag Manager.', 'hakoniwa-tools' ),
			)
		);

	}

	/**
	 * 入力値のサニタイズ
	 *
	 * @param array $input Contains all settings fields as array keys
	 */
	public function sanitize( $input ) {
		$new_input = array();

		if ( isset( $input ) ) {
			foreach ( $input as $key => $val ) {
				// 配列の場合
				if ( is_array( $input[ $key ] ) ) {
					$new_input[ $key ] = array_map( 'absint', $input[ $key ] );
				} else {
					// 通常の場合
					if( $key == 'analytics_gtm_header' || $key == 'analytics_gtm_body' ){
						$new_input[ $key ] = stripslashes( $input[ $key ] );						
					} else {
						$new_input[ $key ] = sanitize_text_field( $input[ $key ] );
					}
				}
			}
		}

		return $new_input;
	}

	/**
	 * GTM(Header)
	 *
	 * @return void
	 */
	public function add_gtm_header() {
		$options = get_option( $this->page_name() );

		if( empty( $options['analytics_range'] ) ){
			if( is_user_logged_in() ){
				return false;
			}
		}

		if ( empty( $options ) ) {
			return false;
		}

		if( ! empty( $options['analytics_gtm'] ) ){
			$code = $options['analytics_gtm'];

			$output = <<< EOM
			<!-- Google Tag Manager -->
			<script>(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
			new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
			j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
			'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
			})(window,document,'script','dataLayer','{$code}');</script>
			<!-- End Google Tag Manager -->
			EOM;

			echo $output . "\n";
		}

	}

	/**
	 * GTM(Body)
	 *
	 * @return void
	 */
	public function add_gtm_body() {

		$options = get_option( $this->page_name() );

		if( empty( $options['analytics_range'] ) ){
			if( is_user_logged_in() ){
				return false;
			}
		}

		if ( empty( $options ) ) {
			return false;
		}

		if( ! empty( $options['analytics_gtm'] ) ){
			$code = $options['analytics_gtm'];

			$output = <<< EOM
			<!-- Google Tag Manager (noscript) -->
			<noscript><iframe src="https://www.googletagmanager.com/ns.html?id='{$code}'" height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>
			<!-- End Google Tag Manager (noscript) -->
			EOM;

			echo $output . "\n";
		}

	}

	/**
	 * Google アナリティクス ID（UA）
	 *
	 * @return void
	 */
	public function add_analytics_tag() {

		$options = get_option( $this->page_name() );

		if ( empty( $options ) ) {
			return false;
		}

		if ( is_admin() || is_user_logged_in() ) {
			return false;
		}

		if( ! empty( $options['analytics_gtag'] ) ){
			$code = $options['analytics_gtag'];

			$output = <<< EOM
				window.dataLayer = window.dataLayer || [];
				function gtag(){dataLayer.push(arguments);}
				gtag('js', new Date());
				gtag('config', '{$code}');
EOM;
			echo '<!-- Global site tag (gtag.js) - Google Analytics -->';
			echo '<script async src="https://www.googletagmanager.com/gtag/js?id=' . esc_attr( $code ) . '"></script>' . "\n";
			echo '<script>' . wp_kses_post( $output ) . '</script>' . "\n";
		}

	}
}

use hakoniwa\tools\options;
new Analytics();
